<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'Staff Management';
$admin = getCurrentAdmin();

$pdo = getDBConnection();

// Get active SOS count for header
$sosCount = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

// Handle form submissions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'add':
                    $name = sanitize($_POST['name']);
                    $email = sanitize($_POST['email']);
                    $phone = sanitize($_POST['phone']);
                    $role = sanitize($_POST['role']);
                    $password = password_hash('password123', PASSWORD_DEFAULT);
                    
                    $stmt = $pdo->prepare("INSERT INTO users (name, email, phone, password, role) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$name, $email, $phone, $password, $role]);
                    
                    $message = 'Staff member added successfully!';
                    $messageType = 'success';
                    logAdminAction('add_staff', "Added new staff: $name");
                    break;
                    
                case 'update_status':
                    $userId = (int)$_POST['user_id'];
                    $status = sanitize($_POST['status']);
                    
                    $stmt = $pdo->prepare("UPDATE users SET status = ? WHERE id = ?");
                    $stmt->execute([$status, $userId]);
                    
                    $message = 'Staff status updated!';
                    $messageType = 'success';
                    logAdminAction('update_staff_status', "Updated status to $status for user ID: $userId");
                    break;
                    
                case 'toggle_duty':
                    $userId = (int)$_POST['user_id'];
                    
                    $stmt = $pdo->prepare("UPDATE users SET is_on_duty = NOT is_on_duty, last_seen = NOW() WHERE id = ?");
                    $stmt->execute([$userId]);
                    
                    $message = 'Duty status toggled!';
                    $messageType = 'success';
                    break;
                    
                case 'delete':
                    $userId = (int)$_POST['user_id'];
                    
                    $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                    $stmt->execute([$userId]);
                    
                    $message = 'Staff member removed!';
                    $messageType = 'success';
                    logAdminAction('delete_staff', "Deleted user ID: $userId");
                    break;
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $messageType = 'error';
        }
    }
}

// Get all staff members
$search = sanitize($_GET['search'] ?? '');
$statusFilter = sanitize($_GET['status'] ?? '');

$query = "SELECT * FROM users WHERE 1=1";
$params = [];

if ($search) {
    $query .= " AND (name LIKE ? OR email LIKE ? OR phone LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

if ($statusFilter) {
    $query .= " AND status = ?";
    $params[] = $statusFilter;
}

$query .= " ORDER BY name ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$staff = $stmt->fetchAll();

// Get staff statistics
$stats = [
    'total' => $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn(),
    'active' => $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn(),
    'on_duty' => $pdo->query("SELECT COUNT(*) FROM users WHERE is_on_duty = 1 AND status = 'active'")->fetchColumn(),
    'inactive' => $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'inactive'")->fetchColumn()
];

include 'includes/header.php';
?>

<!-- Staff Management -->
<div class="fade-in">
    <?php if ($message): ?>
    <div class="mb-6 p-4 rounded-lg flex items-center gap-2 <?php echo $messageType === 'success' ? 'bg-green-50 text-green-700 border border-green-200' : 'bg-red-50 text-red-700 border border-red-200'; ?>">
        <i class="fas fa-<?php echo $messageType === 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i>
        <?php echo htmlspecialchars($message); ?>
    </div>
    <?php endif; ?>

    <!-- Stats -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Total Staff</p>
            <p class="text-2xl font-bold text-gray-800"><?php echo $stats['total']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Active</p>
            <p class="text-2xl font-bold text-green-600"><?php echo $stats['active']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">On Duty</p>
            <p class="text-2xl font-bold text-primary"><?php echo $stats['on_duty']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Inactive</p>
            <p class="text-2xl font-bold text-gray-600"><?php echo $stats['inactive']; ?></p>
        </div>
    </div>

    <!-- Staff Table -->
    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between mb-4 flex-wrap gap-4">
            <h3 class="text-lg font-semibold text-gray-800">All Staff</h3>
            
            <div class="flex items-center gap-3">
                <!-- Search -->
                <form method="GET" class="flex items-center gap-2">
                    <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" 
                        placeholder="Search staff..." 
                        class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary">
                    <select name="status" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary">
                        <option value="">All Status</option>
                        <option value="active" <?php echo $statusFilter === 'active' ? 'selected' : ''; ?>>Active</option>
                        <option value="inactive" <?php echo $statusFilter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        <option value="suspended" <?php echo $statusFilter === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                    </select>
                    <button type="submit" class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200">
                        <i class="fas fa-search"></i>
                    </button>
                </form>
                
                <!-- Add Button -->
                <button onclick="openModal('addStaffModal')" class="px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary-dark">
                    <i class="fas fa-plus mr-2"></i>Add Staff
                </button>
            </div>
        </div>

        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b">
                        <th class="text-left py-3 text-gray-500 font-medium">Name</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Email</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Phone</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Role</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Status</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Duty</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Last Seen</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($staff) > 0): ?>
                        <?php foreach ($staff as $member): ?>
                        <tr class="border-b hover:bg-gray-50">
                            <td class="py-3">
                                <div class="flex items-center gap-3">
                                    <div class="w-8 h-8 rounded-full bg-primary text-white flex items-center justify-center text-sm font-semibold">
                                        <?php echo strtoupper(substr($member['name'], 0, 2)); ?>
                                    </div>
                                    <span class="font-medium"><?php echo htmlspecialchars($member['name']); ?></span>
                                </div>
                            </td>
                            <td class="py-3 text-gray-600"><?php echo htmlspecialchars($member['email']); ?></td>
                            <td class="py-3 text-gray-600"><?php echo htmlspecialchars($member['phone'] ?? '-'); ?></td>
                            <td class="py-3">
                                <span class="px-2 py-1 bg-blue-100 text-blue-700 rounded-full text-xs">
                                    <?php echo ucfirst($member['role']); ?>
                                </span>
                            </td>
                            <td class="py-3">
                                <?php if ($member['status'] === 'active'): ?>
                                <span class="px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs">Active</span>
                                <?php elseif ($member['status'] === 'inactive'): ?>
                                <span class="px-2 py-1 bg-gray-100 text-gray-700 rounded-full text-xs">Inactive</span>
                                <?php else: ?>
                                <span class="px-2 py-1 bg-red-100 text-red-700 rounded-full text-xs">Suspended</span>
                                <?php endif; ?>
                            </td>
                            <td class="py-3">
                                <?php if ($member['is_on_duty']): ?>
                                <span class="text-green-600"><i class="fas fa-circle text-xs mr-1"></i>On Duty</span>
                                <?php else: ?>
                                <span class="text-gray-400"><i class="fas fa-circle text-xs mr-1"></i>Off Duty</span>
                                <?php endif; ?>
                            </td>
                            <td class="py-3 text-gray-500 text-sm">
                                <?php echo $member['last_seen'] ? timeAgo($member['last_seen']) : 'Never'; ?>
                            </td>
                            <td class="py-3">
                                <div class="flex items-center gap-2">
                                    <!-- Toggle Duty -->
                                    <form method="POST" class="inline">
                                        <input type="hidden" name="action" value="toggle_duty">
                                        <input type="hidden" name="user_id" value="<?php echo $member['id']; ?>">
                                        <button type="submit" class="p-2 text-gray-500 hover:text-primary" title="Toggle Duty">
                                            <i class="fas fa-toggle-on"></i>
                                        </button>
                                    </form>
                                    
                                    <!-- View on Map -->
                                    <a href="map.php?focus=<?php echo $member['id']; ?>" class="p-2 text-gray-500 hover:text-primary" title="View on Map">
                                        <i class="fas fa-map-marker-alt"></i>
                                    </a>
                                    
                                    <!-- View Activity -->
                                    <a href="reports.php?staff=<?php echo $member['id']; ?>" class="p-2 text-gray-500 hover:text-primary" title="View Activity">
                                        <i class="fas fa-chart-line"></i>
                                    </a>
                                    
                                    <!-- Delete -->
                                    <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this staff member?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="user_id" value="<?php echo $member['id']; ?>">
                                        <button type="submit" class="p-2 text-gray-500 hover:text-red-600" title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="8" class="py-8 text-center text-gray-500">
                                No staff members found
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add Staff Modal -->
<div id="addStaffModal" class="fixed inset-0 bg-black/50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl w-full max-w-md mx-4">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">Add New Staff</h3>
            <button onclick="closeModal('addStaffModal')" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form method="POST" class="p-4">
            <input type="hidden" name="action" value="add">
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Full Name</label>
                <input type="text" name="name" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary"
                    placeholder="Enter full name">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                <input type="email" name="email" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary"
                    placeholder="Enter email address">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                <input type="tel" name="phone"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary"
                    placeholder="Enter phone number">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">Role</label>
                <select name="role" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary">
                    <option value="guard">Guard</option>
                    <option value="supervisor">Supervisor</option>
                    <option value="admin">Admin</option>
                </select>
            </div>
            
            <div class="flex gap-3">
                <button type="button" onclick="closeModal('addStaffModal')" 
                    class="flex-1 px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary-dark">
                    Add Staff
                </button>
            </div>
        </form>
    </div>
</div>

<script>
    function openModal(id) {
        document.getElementById(id).classList.remove('hidden');
    }
    
    function closeModal(id) {
        document.getElementById(id).classList.add('hidden');
    }
    
    // Close modal on outside click
    document.querySelectorAll('.fixed').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });
    });
</script>

<?php
function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
