<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'Reports';
$admin = getCurrentAdmin();

$pdo = getDBConnection();

// Get active SOS count for header
$sosCount = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

// Get date filters
$dateRange = sanitize($_GET['range'] ?? 'today');
$startDate = '';
$endDate = date('Y-m-d');

switch ($dateRange) {
    case 'today':
        $startDate = date('Y-m-d');
        break;
    case 'week':
        $startDate = date('Y-m-d', strtotime('-7 days'));
        break;
    case 'month':
        $startDate = date('Y-m-d', strtotime('-30 days'));
        break;
    case 'custom':
        $startDate = sanitize($_GET['start'] ?? date('Y-m-d', strtotime('-30 days')));
        $endDate = sanitize($_GET['end'] ?? date('Y-m-d'));
        break;
}

// Get statistics based on date range
$checkinStats = $pdo->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'verified' THEN 1 ELSE 0 END) as verified,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
    FROM check_ins 
    WHERE DATE(created_at) BETWEEN ? AND ?
");
$checkinStats->execute([$startDate, $endDate]);
$checkinData = $checkinStats->fetch();

$sosStats = $pdo->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
        SUM(CASE WHEN status = 'acknowledged' THEN 1 ELSE 0 END) as acknowledged,
        SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved,
        SUM(CASE WHEN status = 'false_alarm' THEN 1 ELSE 0 END) as false_alarm
    FROM sos_alerts 
    WHERE DATE(created_at) BETWEEN ? AND ?
");
$sosStats->execute([$startDate, $endDate]);
$sosData = $sosStats->fetch();

// Get daily check-ins for chart
$dailyCheckins = $pdo->prepare("
    SELECT DATE(created_at) as date, COUNT(*) as count 
    FROM check_ins 
    WHERE DATE(created_at) BETWEEN ? AND ?
    GROUP BY DATE(created_at)
    ORDER BY date
");
$dailyCheckins->execute([$startDate, $endDate]);
$dailyCheckinData = $dailyCheckins->fetchAll(PDO::FETCH_KEY_PAIR);

// Get staff activity
$staffActivity = $pdo->prepare("
    SELECT 
        u.id,
        u.name,
        COUNT(DISTINCT c.id) as checkin_count,
        COUNT(DISTINCT s.id) as sos_count,
        MAX(c.created_at) as last_checkin,
        MAX(s.created_at) as last_sos
    FROM users u
    LEFT JOIN check_ins c ON c.user_id = u.id AND DATE(c.created_at) BETWEEN ? AND ?
    LEFT JOIN sos_alerts s ON s.user_id = u.id AND DATE(s.created_at) BETWEEN ? AND ?
    WHERE u.status = 'active'
    GROUP BY u.id
    ORDER BY checkin_count DESC
    LIMIT 10
");
$staffActivity->execute([$startDate, $endDate, $startDate, $endDate]);
$staffData = $staffActivity->fetchAll();

// Get hourly distribution
$hourlyDistribution = $pdo->prepare("
    SELECT HOUR(created_at) as hour, COUNT(*) as count 
    FROM check_ins 
    WHERE DATE(created_at) BETWEEN ? AND ?
    GROUP BY HOUR(created_at)
    ORDER BY hour
");
$hourlyDistribution->execute([$startDate, $endDate]);
$hourlyData = $hourlyDistribution->fetchAll(PDO::FETCH_KEY_PAIR);

// Format hourly data for chart
$hourlyChartData = [];
for ($i = 0; $i < 24; $i++) {
    $hourlyChartData[$i] = $hourlyData[$i] ?? 0;
}

include 'includes/header.php';
?>

<!-- Reports -->
<div class="fade-in">
    <!-- Date Range Selector -->
    <div class="bg-white rounded-xl p-4 shadow-sm mb-6">
        <form method="GET" class="flex flex-wrap items-center gap-4">
            <div class="flex items-center gap-2">
                <label class="text-sm text-gray-600">Date Range:</label>
                <select name="range" onchange="this.form.submit()" class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
                    <option value="today" <?php echo $dateRange === 'today' ? 'selected' : ''; ?>>Today</option>
                    <option value="week" <?php echo $dateRange === 'week' ? 'selected' : ''; ?>>This Week</option>
                    <option value="month" <?php echo $dateRange === 'month' ? 'selected' : ''; ?>>This Month</option>
                    <option value="custom" <?php echo $dateRange === 'custom' ? 'selected' : ''; ?>>Custom Range</option>
                </select>
            </div>
            
            <?php if ($dateRange === 'custom'): ?>
            <div class="flex items-center gap-2">
                <input type="date" name="start" value="<?php echo $startDate; ?>" 
                    class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
                <span class="text-gray-400">to</span>
                <input type="date" name="end" value="<?php echo $endDate; ?>" 
                    class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
                <button type="submit" class="px-4 py-2 bg-primary text-white rounded-lg text-sm">
                    Apply
                </button>
            </div>
            <?php endif; ?>
            
            <div class="ml-auto">
                <span class="text-sm text-gray-500">
                    Showing data from <?php echo date('M d, Y', strtotime($startDate)); ?> to <?php echo date('M d, Y', strtotime($endDate)); ?>
                </span>
            </div>
        </form>
    </div>

    <!-- Summary Stats -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
        <!-- Check-in Stats -->
        <div class="bg-white rounded-xl p-6 shadow-sm">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-800">Total Check-ins</h3>
                <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center">
                    <i class="fas fa-camera text-blue-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-800"><?php echo number_format($checkinData['total']); ?></p>
            <div class="mt-3 flex items-center gap-4 text-sm">
                <span class="text-green-600"><i class="fas fa-check mr-1"></i><?php echo $checkinData['verified']; ?> verified</span>
                <span class="text-yellow-600"><i class="fas fa-clock mr-1"></i><?php echo $checkinData['pending']; ?> pending</span>
            </div>
        </div>

        <div class="bg-white rounded-xl p-6 shadow-sm">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-800">Verification Rate</h3>
                <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center">
                    <i class="fas fa-check-circle text-green-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-800">
                <?php echo $checkinData['total'] > 0 ? round(($checkinData['verified'] / $checkinData['total']) * 100) : 0; ?>%
            </p>
            <p class="text-sm text-gray-500 mt-2">of all check-ins verified</p>
        </div>

        <div class="bg-white rounded-xl p-6 shadow-sm">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-800">SOS Alerts</h3>
                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center">
                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-800"><?php echo number_format($sosData['total']); ?></p>
            <div class="mt-3 flex items-center gap-4 text-sm">
                <span class="text-red-600"><i class="fas fa-circle mr-1" style="font-size: 8px;"></i><?php echo $sosData['active']; ?> active</span>
                <span class="text-green-600"><i class="fas fa-check mr-1"></i><?php echo $sosData['resolved']; ?> resolved</span>
            </div>
        </div>

        <div class="bg-white rounded-xl p-6 shadow-sm">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-800">False Alarms</h3>
                <div class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center">
                    <i class="fas fa-times-circle text-gray-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-800"><?php echo number_format($sosData['false_alarm']); ?></p>
            <p class="text-sm text-gray-500 mt-2">
                <?php echo $sosData['total'] > 0 ? round(($sosData['false_alarm'] / $sosData['total']) * 100) : 0; ?>% of total alerts
            </p>
        </div>
    </div>

    <!-- Charts -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
        <!-- Daily Check-ins Chart -->
        <div class="bg-white rounded-xl p-6 shadow-sm">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Daily Check-ins</h3>
            <canvas id="dailyChart" height="250"></canvas>
        </div>

        <!-- Hourly Distribution Chart -->
        <div class="bg-white rounded-xl p-6 shadow-sm">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Hourly Distribution</h3>
            <canvas id="hourlyChart" height="250"></canvas>
        </div>
    </div>

    <!-- Staff Activity Table -->
    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-800">Top Staff Activity</h3>
            <button onclick="exportReport()" class="px-4 py-2 bg-primary text-white rounded-lg text-sm hover:bg-primary-dark">
                <i class="fas fa-download mr-2"></i>Export CSV
            </button>
        </div>
        
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b">
                        <th class="text-left py-3 text-gray-500 font-medium">Staff</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Check-ins</th>
                        <th class="text-left py-3 text-gray-500 font-medium">SOS Alerts</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Last Check-in</th>
                        <th class="text-left py-3 text-gray-500 font-medium">Last SOS</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($staffData) > 0): ?>
                        <?php foreach ($staffData as $staff): ?>
                        <tr class="border-b hover:bg-gray-50">
                            <td class="py-3">
                                <div class="flex items-center gap-3">
                                    <div class="w-8 h-8 rounded-full bg-primary text-white flex items-center justify-center text-sm font-semibold">
                                        <?php echo strtoupper(substr($staff['name'], 0, 2)); ?>
                                    </div>
                                    <span class="font-medium"><?php echo htmlspecialchars($staff['name']); ?></span>
                                </div>
                            </td>
                            <td class="py-3">
                                <span class="px-2 py-1 bg-green-100 text-green-700 rounded-full text-sm">
                                    <?php echo $staff['checkin_count']; ?>
                                </span>
                            </td>
                            <td class="py-3">
                                <?php if ($staff['sos_count'] > 0): ?>
                                <span class="px-2 py-1 bg-red-100 text-red-700 rounded-full text-sm">
                                    <?php echo $staff['sos_count']; ?>
                                </span>
                                <?php else: ?>
                                <span class="text-gray-400">0</span>
                                <?php endif; ?>
                            </td>
                            <td class="py-3 text-gray-600">
                                <?php echo $staff['last_checkin'] ? timeAgo($staff['last_checkin']) : 'Never'; ?>
                            </td>
                            <td class="py-3 text-gray-600">
                                <?php echo $staff['last_sos'] ? timeAgo($staff['last_sos']) : 'Never'; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="5" class="py-8 text-center text-gray-500">
                                No activity data for this period
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Export Modal -->
<div id="exportModal" class="fixed inset-0 bg-black/50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl w-full max-w-md mx-4">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">Export Report</h3>
            <button onclick="closeModal('exportModal')" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="p-4">
            <p class="text-gray-600 mb-4">Select export format:</p>
            <div class="space-y-2">
                <a href="api/export.php?type=report&format=csv&start=<?php echo $startDate; ?>&end=<?php echo $endDate; ?>" 
                    class="flex items-center gap-3 p-3 border rounded-lg hover:bg-gray-50">
                    <i class="fas fa-file-csv text-green-600 text-xl"></i>
                    <span>CSV (Excel)</span>
                </a>
                <a href="api/export.php?type=report&format=json&start=<?php echo $startDate; ?>&end=<?php echo $endDate; ?>" 
                    class="flex items-center gap-3 p-3 border rounded-lg hover:bg-gray-50">
                    <i class="fas fa-file-code text-blue-600 text-xl"></i>
                    <span>JSON</span>
                </a>
                <a href="api/export.php?type=report&format=pdf&start=<?php echo $startDate; ?>&end=<?php echo $endDate; ?>" 
                    class="flex items-center gap-3 p-3 border rounded-lg hover:bg-gray-50">
                    <i class="fas fa-file-pdf text-red-600 text-xl"></i>
                    <span>PDF</span>
                </a>
            </div>
        </div>
    </div>
</div>

<script>
    // Daily Check-ins Chart
    const dailyCtx = document.getElementById('dailyChart');
    if (dailyCtx) {
        new Chart(dailyCtx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: <?php echo json_encode(array_keys($dailyCheckinData)); ?>,
                datasets: [{
                    label: 'Check-ins',
                    data: <?php echo json_encode(array_values($dailyCheckinData)); ?>,
                    backgroundColor: '#fc8100',
                    borderRadius: 4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }

    // Hourly Distribution Chart
    const hourlyCtx = document.getElementById('hourlyChart');
    if (hourlyCtx) {
        new Chart(hourlyCtx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '21', '22', '23'],
                datasets: [{
                    label: 'Check-ins',
                    data: <?php echo json_encode(array_values($hourlyChartData)); ?>,
                    borderColor: '#1a217f',
                    backgroundColor: 'rgba(26, 33, 127, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Hour of Day'
                        }
                    }
                }
            }
        });
    }

    function exportReport() {
        openModal('exportModal');
    }
    
    function openModal(id) {
        document.getElementById(id).classList.remove('hidden');
    }
    
    function closeModal(id) {
        document.getElementById(id).classList.add('hidden');
    }
    
    // Close modal on outside click
    document.querySelectorAll('.fixed').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });
    });
</script>

<?php
function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
