<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'Dashboard Overview';
$admin = getCurrentAdmin();

// Get dashboard statistics
$pdo = getDBConnection();

// Total staff count
$totalStaff = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn();

// Staff on duty now
$onDuty = $pdo->query("SELECT COUNT(*) FROM users WHERE is_on_duty = 1 AND status = 'active'")->fetchColumn();

// Check-ins today
$todayCheckins = $pdo->query("SELECT COUNT(*) FROM check_ins WHERE DATE(created_at) = CURDATE()")->fetchColumn();

// Active SOS alerts
$activeSOS = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

// Get recent activity
$recentActivity = $pdo->query("
    SELECT 
        CASE 
            WHEN c.id IS NOT NULL THEN 'checkin'
            WHEN s.id IS NOT NULL THEN 'sos'
            ELSE 'location'
        END as type,
        COALESCE(c.created_at, s.created_at, l.created_at) as time,
        u.name as user_name,
        COALESCE(c.location_name, s.location_name, l.location_name) as location,
        CASE 
            WHEN c.id IS NOT NULL THEN 'Check-in'
            WHEN s.id IS NOT NULL THEN 'SOS Alert'
            ELSE 'Location Update'
        END as action,
        CASE 
            WHEN c.status = 'verified' THEN 'verified'
            WHEN s.status = 'active' THEN 'active'
            ELSE 'normal'
        END as status
    FROM users u
    LEFT JOIN check_ins c ON c.user_id = u.id
    LEFT JOIN sos_alerts s ON s.user_id = u.id
    LEFT JOIN (
        SELECT user_id, latitude, longitude, location_name, created_at,
               ROW_NUMBER() OVER (PARTITION BY user_id ORDER BY created_at DESC) as rn
        FROM location_logs
    ) l ON l.user_id = u.id AND l.rn = 1
    WHERE c.id IS NOT NULL OR s.id IS NOT NULL OR l.user_id IS NOT NULL
    ORDER BY time DESC
    LIMIT 10
")->fetchAll();

// Get chart data - hourly check-ins today
$hourlyCheckins = $pdo->query("
    SELECT HOUR(created_at) as hour, COUNT(*) as count 
    FROM check_ins 
    WHERE DATE(created_at) = CURDATE() 
    GROUP BY HOUR(created_at)
    ORDER BY hour
")->fetchAll(PDO::FETCH_KEY_PAIR);

// Get check-in status breakdown
$checkinStats = $pdo->query("
    SELECT status, COUNT(*) as count 
    FROM check_ins 
    WHERE DATE(created_at) = CURDATE() 
    GROUP BY status
")->fetchAll(PDO::FETCH_KEY_PAIR);

// Fill missing hours with 0
$hourlyData = [];
for ($i = 6; $i <= 20; $i++) {
    $hourlyData[$i] = $hourlyCheckins[$i] ?? 0;
}

// Include header
include 'includes/header.php';
?>

<!-- Stats Cards -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8 fade-in">
    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Total Staff</p>
                <p class="text-3xl font-bold text-gray-800"><?php echo number_format($totalStaff); ?></p>
            </div>
            <div class="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center">
                <i class="fas fa-users text-primary text-xl"></i>
            </div>
        </div>
        <p class="text-green-500 text-sm mt-2">
            <i class="fas fa-arrow-up"></i> Active guards
        </p>
    </div>

    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">On Duty Now</p>
                <p class="text-3xl font-bold text-gray-800"><?php echo number_format($onDuty); ?></p>
            </div>
            <div class="w-12 h-12 rounded-full bg-green-100 flex items-center justify-center">
                <i class="fas fa-user-check text-status-safe text-xl"></i>
            </div>
        </div>
        <p class="text-gray-500 text-sm mt-2"><?php echo $totalStaff > 0 ? round(($onDuty / $totalStaff) * 100) : 0; ?>% active</p>
    </div>

    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Check-ins Today</p>
                <p class="text-3xl font-bold text-gray-800"><?php echo number_format($todayCheckins); ?></p>
            </div>
            <div class="w-12 h-12 rounded-full bg-purple-100 flex items-center justify-center">
                <i class="fas fa-camera text-purple-600 text-xl"></i>
            </div>
        </div>
        <p class="text-green-500 text-sm mt-2">
            <i class="fas fa-clock"></i> Real-time data
        </p>
    </div>

    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Active SOS</p>
                <p class="text-3xl font-bold text-status-danger"><?php echo number_format($activeSOS); ?></p>
            </div>
            <div class="w-12 h-12 rounded-full bg-red-100 flex items-center justify-center">
                <i class="fas fa-exclamation-triangle text-status-danger text-xl"></i>
            </div>
        </div>
        <p class="text-status-danger text-sm mt-2">
            <?php echo $activeSOS > 0 ? 'Requires attention' : 'All clear'; ?>
        </p>
    </div>
</div>

<!-- Charts Row -->
<div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8 fade-in">
    <div class="bg-white rounded-xl p-6 shadow-sm">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Staff Activity Today</h3>
        <canvas id="activityChart" height="200"></canvas>
    </div>

    <div class="bg-white rounded-xl p-6 shadow-sm">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Check-in Status</h3>
        <canvas id="checkinChart" height="200"></canvas>
    </div>
</div>

<!-- Recent Activity Table -->
<div class="bg-white rounded-xl p-6 shadow-sm fade-in">
    <div class="flex items-center justify-between mb-4">
        <h3 class="text-lg font-semibold text-gray-800">Recent Activity</h3>
        <a href="reports.php" class="text-primary hover:underline text-sm">View All</a>
    </div>
    <table class="w-full">
        <thead>
            <tr class="border-b">
                <th class="text-left py-3 text-gray-500 font-medium">Staff</th>
                <th class="text-left py-3 text-gray-500 font-medium">Action</th>
                <th class="text-left py-3 text-gray-500 font-medium">Location</th>
                <th class="text-left py-3 text-gray-500 font-medium">Time</th>
                <th class="text-left py-3 text-gray-500 font-medium">Status</th>
            </tr>
        </thead>
        <tbody>
            <?php if (count($recentActivity) > 0): ?>
                <?php foreach ($recentActivity as $activity): ?>
                <tr class="border-b hover:bg-gray-50">
                    <td class="py-3">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full bg-primary text-white flex items-center justify-center text-sm">
                                <?php echo strtoupper(substr($activity['user_name'], 0, 2)); ?>
                            </div>
                            <span><?php echo htmlspecialchars($activity['user_name']); ?></span>
                        </div>
                    </td>
                    <td class="py-3"><?php echo htmlspecialchars($activity['action']); ?></td>
                    <td class="py-3"><?php echo htmlspecialchars($activity['location'] ?? 'Unknown'); ?></td>
                    <td class="py-3"><?php echo timeAgo($activity['time']); ?></td>
                    <td class="py-3">
                        <?php if ($activity['status'] === 'verified'): ?>
                        <span class="px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs">Verified</span>
                        <?php elseif ($activity['status'] === 'active'): ?>
                        <span class="px-2 py-1 bg-red-100 text-red-700 rounded-full text-xs">Active</span>
                        <?php else: ?>
                        <span class="px-2 py-1 bg-gray-100 text-gray-700 rounded-full text-xs">Normal</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td colspan="5" class="py-8 text-center text-gray-500">No recent activity</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<script>
    // Initialize Charts with dynamic data
    document.addEventListener('DOMContentLoaded', function() {
        // Activity Chart
        const activityCtx = document.getElementById('activityChart');
        if (activityCtx) {
            new Chart(activityCtx.getContext('2d'), {
                type: 'line',
                data: {
                    labels: ['6am', '7am', '8am', '9am', '10am', '11am', '12pm', '1pm', '2pm', '3pm', '4pm', '5pm', '6pm', '7pm', '8pm'],
                    datasets: [{
                        label: 'Check-ins',
                        data: [<?php echo implode(',', array_values($hourlyData)); ?>],
                        borderColor: '#fc8100',
                        backgroundColor: 'linear-gradient(180deg, rgba(252,129,0,0.3) 0%, rgba(252,129,0,0.05) 100%)',
                        borderWidth: 3,
                        tension: 0.4,
                        fill: true,
                        pointBackgroundColor: '#fc8100',
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: '#fc8100',
                            titleColor: '#fff',
                            bodyColor: '#fff',
                            padding: 12,
                            cornerRadius: 8
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(0,0,0,0.05)'
                            },
                            ticks: {
                                stepSize: 1
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        }

        // Check-in Status Chart
        const checkinCtx = document.getElementById('checkinChart');
        if (checkinCtx) {
            new Chart(checkinCtx.getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: ['Verified', 'Pending', 'Failed'],
                    datasets: [{
                        data: [
                            <?php echo $checkinStats['verified'] ?? 0; ?>,
                            <?php echo $checkinStats['pending'] ?? 0; ?>,
                            <?php echo $checkinStats['failed'] ?? 0; ?>
                        ],
                        backgroundColor: ['#22c55e', '#f59e0b', '#ef4444'],
                        borderColor: ['#ffffff', '#ffffff', '#ffffff'],
                        borderWidth: 3,
                        hoverOffset: 8
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '65%',
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 20,
                                usePointStyle: true,
                                pointStyle: 'circle'
                            }
                        },
                        tooltip: {
                            backgroundColor: '#1a217f',
                            titleColor: '#fff',
                            bodyColor: '#fff',
                            padding: 12,
                            cornerRadius: 8
                        }
                    }
                }
            });
        }
    });
</script>

<?php
// Helper function for time ago
function timeAgo($datetime) {
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
