<?php
require_once 'includes/config.php';
requireLogin();

$pageTitle = 'Photo Evidence';
$admin = getCurrentAdmin();

$pdo = getDBConnection();

// Get active SOS count for header
$sosCount = $pdo->query("SELECT COUNT(*) FROM sos_alerts WHERE status = 'active'")->fetchColumn();

// Handle form submissions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'verify':
                    $checkinId = (int)$_POST['checkin_id'];
                    $notes = sanitize($_POST['notes'] ?? '');
                    
                    $stmt = $pdo->prepare("UPDATE check_ins SET status = 'verified', verified_by = ?, verified_at = NOW(), notes = CONCAT(IFNULL(notes, ''), ?)");
                    $stmt->execute([$admin['id'], "\n[Verified] " . $notes]);
                    
                    $message = 'Check-in verified successfully!';
                    $messageType = 'success';
                    logAdminAction('verify_checkin', "Verified check-in ID: $checkinId");
                    break;
                    
                case 'reject':
                    $checkinId = (int)$_POST['checkin_id'];
                    $notes = sanitize($_POST['notes'] ?? '');
                    
                    $stmt = $pdo->prepare("UPDATE check_ins SET status = 'failed', verified_by = ?, verified_at = NOW(), notes = CONCAT(IFNULL(notes, ''), ?)");
                    $stmt->execute([$admin['id'], "\n[Rejected] " . $notes]);
                    
                    $message = 'Check-in rejected!';
                    $messageType = 'success';
                    logAdminAction('reject_checkin', "Rejected check-in ID: $checkinId");
                    break;
                    
                case 'delete':
                    $checkinId = (int)$_POST['checkin_id'];
                    
                    $stmt = $pdo->prepare("DELETE FROM check_ins WHERE id = ?");
                    $stmt->execute([$checkinId]);
                    
                    $message = 'Check-in deleted!';
                    $messageType = 'success';
                    break;
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $messageType = 'error';
        }
    }
}

// Get filter parameters
$statusFilter = sanitize($_GET['status'] ?? '');
$dateFilter = sanitize($_GET['date'] ?? '');
$staffFilter = (int)($_GET['staff'] ?? 0);

// Build query
$query = "
    SELECT c.*, u.name as user_name, u.phone as user_phone,
           a.name as verified_by_name
    FROM check_ins c
    JOIN users u ON c.user_id = u.id
    LEFT JOIN admins a ON c.verified_by = a.id
    WHERE 1=1
";

$params = [];

if ($statusFilter) {
    $query .= " AND c.status = ?";
    $params[] = $statusFilter;
}

if ($dateFilter) {
    $query .= " AND DATE(c.created_at) = ?";
    $params[] = $dateFilter;
}

if ($staffFilter) {
    $query .= " AND c.user_id = ?";
    $params[] = $staffFilter;
}

$query .= " ORDER BY c.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$checkins = $stmt->fetchAll();

// Get statistics
$stats = [
    'today' => $pdo->query("SELECT COUNT(*) FROM check_ins WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
    'verified' => $pdo->query("SELECT COUNT(*) FROM check_ins WHERE status = 'verified' AND DATE(created_at) = CURDATE()")->fetchColumn(),
    'pending' => $pdo->query("SELECT COUNT(*) FROM check_ins WHERE status = 'pending'")->fetchColumn(),
    'failed' => $pdo->query("SELECT COUNT(*) FROM check_ins WHERE status = 'failed'")->fetchColumn()
];

// Get staff list for filter
$staffList = $pdo->query("SELECT id, name FROM users WHERE status = 'active' ORDER BY name")->fetchAll();

include 'includes/header.php';
?>

<!-- Check-ins / Photo Evidence -->
<div class="fade-in">
    <?php if ($message): ?>
    <div class="mb-6 p-4 rounded-lg flex items-center gap-2 <?php echo $messageType === 'success' ? 'bg-green-50 text-green-700 border border-green-200' : 'bg-red-50 text-red-700 border border-red-200'; ?>">
        <i class="fas fa-<?php echo $messageType === 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i>
        <?php echo htmlspecialchars($message); ?>
    </div>
    <?php endif; ?>

    <!-- Stats -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Today's Check-ins</p>
            <p class="text-2xl font-bold text-gray-800"><?php echo $stats['today']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Verified</p>
            <p class="text-2xl font-bold text-green-600"><?php echo $stats['verified']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Pending</p>
            <p class="text-2xl font-bold text-yellow-600"><?php echo $stats['pending']; ?></p>
        </div>
        <div class="bg-white rounded-xl p-4 shadow-sm">
            <p class="text-gray-500 text-sm">Failed/Rejected</p>
            <p class="text-2xl font-bold text-red-600"><?php echo $stats['failed']; ?></p>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-xl p-4 shadow-sm mb-6">
        <form method="GET" class="flex flex-wrap items-center gap-4">
            <div class="flex items-center gap-2">
                <label class="text-sm text-gray-600">Status:</label>
                <select name="px-3status" class=" py-2 border border-gray-300 rounded-lg text-sm">
                    <option value="">All</option>
                    <option value="pending" <?php echo $statusFilter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                    <option value="verified" <?php echo $statusFilter === 'verified' ? 'selected' : ''; ?>>Verified</option>
                    <option value="failed" <?php echo $statusFilter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                </select>
            </div>
            
            <div class="flex items-center gap-2">
                <label class="text-sm text-gray-600">Date:</label>
                <input type="date" name="date" value="<?php echo htmlspecialchars($dateFilter); ?>" 
                    class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
            </div>
            
            <div class="flex items-center gap-2">
                <label class="text-sm text-gray-600">Staff:</label>
                <select name="staff" class="px-3 py-2 border border-gray-300 rounded-lg text-sm">
                    <option value="">All Staff</option>
                    <?php foreach ($staffList as $s): ?>
                    <option value="<?php echo $s['id']; ?>" <?php echo $staffFilter === $s['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($s['name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <button type="submit" class="px-4 py-2 bg-primary text-white rounded-lg text-sm hover:bg-primary-dark">
                <i class="fas fa-filter mr-1"></i> Filter
            </button>
            
            <a href="checkins.php" class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg text-sm hover:bg-gray-200">
                <i class="fas fa-times mr-1"></i> Clear
            </a>
        </form>
    </div>

    <!-- Photo Grid -->
    <div class="bg-white rounded-xl p-6 shadow-sm">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-800">Photo Evidence</h3>
            <button onclick="exportCheckins()" class="px-4 py-2 bg-primary text-white rounded-lg text-sm hover:bg-primary-dark">
                <i class="fas fa-download mr-2"></i>Export
            </button>
        </div>
        
        <?php if (count($checkins) > 0): ?>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
            <?php foreach ($checkins as $checkin): ?>
            <div class="border rounded-lg overflow-hidden hover:shadow-md transition-shadow">
                <!-- Photo -->
                <div class="relative h-48 bg-gray-100">
                    <?php if ($checkin['photo_url']): ?>
                    <img src="<?php echo htmlspecialchars($checkin['photo_url']); ?>" 
                        alt="Check-in photo" class="w-full h-full object-cover">
                    <?php else: ?>
                    <div class="w-full h-full flex items-center justify-center">
                        <i class="fas fa-camera text-4xl text-gray-300"></i>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Status Badge -->
                    <div class="absolute top-2 right-2">
                        <?php if ($checkin['status'] === 'verified'): ?>
                        <span class="px-2 py-1 bg-green-500 text-white rounded-full text-xs">
                            <i class="fas fa-check mr-1"></i>Verified
                        </span>
                        <?php elseif ($checkin['status'] === 'pending'): ?>
                        <span class="px-2 py-1 bg-yellow-500 text-white rounded-full text-xs">
                            <i class="fas fa-clock mr-1"></i>Pending
                        </span>
                        <?php else: ?>
                        <span class="px-2 py-1 bg-red-500 text-white rounded-full text-xs">
                            <i class="fas fa-times mr-1"></i>Failed
                        </span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Info -->
                <div class="p-3">
                    <div class="flex items-center gap-2 mb-2">
                        <div class="w-8 h-8 rounded-full bg-primary text-white flex items-center justify-center text-xs font-semibold">
                            <?php echo strtoupper(substr($checkin['user_name'], 0, 2)); ?>
                        </div>
                        <div>
                            <p class="font-medium text-sm"><?php echo htmlspecialchars($checkin['user_name']); ?></p>
                            <p class="text-xs text-gray-500"><?php echo timeAgo($checkin['created_at']); ?></p>
                        </div>
                    </div>
                    
                    <p class="text-sm text-gray-600 mb-2">
                        <i class="fas fa-map-marker-alt mr-1"></i>
                        <?php echo htmlspecialchars($checkin['location_name'] ?? 'Unknown location'); ?>
                    </p>
                    
                    <?php if ($checkin['notes']): ?>
                    <p class="text-xs text-gray-500 mb-2 border-t pt-2">
                        <i class="fas fa-sticky-note mr-1"></i>
                        <?php echo htmlspecialchars($checkin['notes']); ?>
                    </p>
                    <?php endif; ?>
                    
                    <!-- Actions -->
                    <?php if ($checkin['status'] === 'pending'): ?>
                    <div class="flex gap-2 mt-3 pt-2 border-t">
                        <form method="POST" class="flex-1">
                            <input type="hidden" name="action" value="verify">
                            <input type="hidden" name="checkin_id" value="<?php echo $checkin['id']; ?>">
                            <button type="submit" class="w-full px-2 py-1 bg-green-500 text-white rounded text-xs hover:bg-green-600">
                                <i class="fas fa-check mr-1"></i>Verify
                            </button>
                        </form>
                        <form method="POST" class="flex-1">
                            <input type="hidden" name="action" value="reject">
                            <input type="hidden" name="checkin_id" value="<?php echo $checkin['id']; ?>">
                            <button type="submit" class="w-full px-2 py-1 bg-red-500 text-white rounded text-xs hover:bg-red-600">
                                <i class="fas fa-times mr-1"></i>Reject
                            </button>
                        </form>
                    </div>
                    <?php else: ?>
                    <div class="mt-3 pt-2 border-t text-xs text-gray-500">
                        <?php if ($checkin['verified_by_name']): ?>
                        <p><i class="fas fa-user-check mr-1"></i>By: <?php echo htmlspecialchars($checkin['verified_by_name']); ?></p>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php else: ?>
        <div class="text-center py-12">
            <i class="fas fa-camera-slash text-5xl text-gray-300 mb-4"></i>
            <p class="text-gray-500">No check-ins found</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Export Modal -->
<div id="exportModal" class="fixed inset-0 bg-black/50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl w-full max-w-md mx-4">
        <div class="p-4 border-b flex items-center justify-between">
            <h3 class="text-lg font-semibold">Export Check-ins</h3>
            <button onclick="closeModal('exportModal')" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="p-4">
            <p class="text-gray-600 mb-4">Select export format:</p>
            <div class="space-y-2">
                <a href="api/export.php?type=checkins&format=csv" class="flex items-center gap-3 p-3 border rounded-lg hover:bg-gray-50">
                    <i class="fas fa-file-csv text-green-600 text-xl"></i>
                    <span>CSV (Excel)</span>
                </a>
                <a href="api/export.php?type=checkins&format=json" class="flex items-center gap-3 p-3 border rounded-lg hover:bg-gray-50">
                    <i class="fas fa-file-code text-blue-600 text-xl"></i>
                    <span>JSON</span>
                </a>
            </div>
        </div>
    </div>
</div>

<script>
    function exportCheckins() {
        openModal('exportModal');
    }
    
    function openModal(id) {
        document.getElementById(id).classList.remove('hidden');
    }
    
    function closeModal(id) {
        document.getElementById(id).classList.add('hidden');
    }
    
    // Close modal on outside click
    document.querySelectorAll('.fixed').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });
    });
</script>

<?php
function timeAgo($datetime) {
    if (!$datetime) return 'Never';
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } else {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    }
}

include 'includes/footer.php';
