<?php
/**
 * Mobile API - For Android App Communication
 * Handles authentication, check-ins, location updates, and SOS alerts
 */

require_once '../includes/config.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    $pdo = getDBConnection();
    $action = $_GET['action'] ?? '';
    
    // Public endpoints (no auth required)
    $publicActions = ['login', 'register'];
    
    // Check authentication for protected endpoints
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $userId = null;
    
    if (!in_array($action, $publicActions)) {
        if (empty($token)) {
            http_response_code(401);
            echo json_encode(['success' => false, 'message' => 'Unauthorized']);
            exit;
        }
        
        // Validate token (simple implementation - in production use JWT)
        $userId = validateToken($pdo, $token);
        if (!$userId) {
            http_response_code(401);
            echo json_encode(['success' => false, 'message' => 'Invalid token']);
            exit;
        }
    }
    
    switch ($action) {
        // ==================== AUTH ====================
        case 'login':
            login($pdo);
            break;
            
        case 'register':
            register($pdo);
            break;
            
        case 'logout':
            logout($pdo, $userId);
            break;
            
        // ==================== PROFILE ====================
        case 'profile':
            getProfile($pdo, $userId);
            break;
            
        case 'update_profile':
            updateProfile($pdo, $userId);
            break;
            
        // ==================== CHECK-INS ====================
        case 'checkin':
            createCheckin($pdo, $userId);
            break;
            
        case 'my_checkins':
            getMyCheckins($pdo, $userId);
            break;
            
        // ==================== LOCATIONS ====================
        case 'update_location':
            updateLocation($pdo, $userId);
            break;
            
        case 'my_location':
            getMyLocation($pdo, $userId);
            break;
            
        // ==================== SOS ====================
        case 'sos':
            createSOS($pdo, $userId);
            break;
            
        case 'my_sos':
            getMySOS($pdo, $userId);
            break;
            
        case 'resolve_sos':
            resolveSOS($pdo, $userId);
            break;
            
        // ==================== STAFF ====================
        case 'all_staff':
            getAllStaff($pdo, $userId);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

// ==================== FUNCTIONS ====================

function validateToken($pdo, $token) {
    // Simple token validation - in production, use JWT or proper session management
    $stmt = $pdo->prepare("SELECT id FROM users WHERE fcm_token = ? AND status = 'active' LIMIT 1");
    $stmt->execute([$token]);
    $user = $stmt->fetch();
    return $user ? $user['id'] : null;
}

function generateToken($userId) {
    return bin2hex(random_bytes(32)) . '.' . $userId;
}

function login($pdo) {
    $email = sanitize($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Email and password required']);
        return;
    }
    
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND status = 'active'");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if ($user && password_verify($password, $user['password'])) {
        // Generate token
        $token = generateToken($user['id']);
        
        // Update FCM token
        $updateStmt = $pdo->prepare("UPDATE users SET fcm_token = ? WHERE id = ?");
        $updateStmt->execute([$token, $user['id']]);
        
        // Update last seen
        $pdo->prepare("UPDATE users SET last_seen = NOW() WHERE id = ?")->execute([$user['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'user' => [
                    'id' => $user['id'],
                    'name' => $user['name'],
                    'email' => $user['email'],
                    'phone' => $user['phone'],
                    'role' => $user['role']
                ],
                'token' => $token
            ]
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
    }
}

function register($pdo) {
    $name = sanitize($_POST['name'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $phone = sanitize($_POST['phone'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($name) || empty($email) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Name, email and password required']);
        return;
    }
    
    // Check if email exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Email already registered']);
        return;
    }
    
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $pdo->prepare("INSERT INTO users (name, email, phone, password, role) VALUES (?, ?, ?, ?, 'guard')");
    $stmt->execute([$name, $email, $phone, $hashedPassword]);
    
    $userId = $pdo->lastInsertId();
    $token = generateToken($userId);
    
    // Update token
    $pdo->prepare("UPDATE users SET fcm_token = ? WHERE id = ?")->execute([$token, $userId]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Registration successful',
        'data' => [
            'user' => [
                'id' => $userId,
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'role' => 'guard'
            ],
            'token' => $token
        ]
    ]);
}

function logout($pdo, $userId) {
    $pdo->prepare("UPDATE users SET fcm_token = NULL WHERE id = ?")->execute([$userId]);
    echo json_encode(['success' => true, 'message' => 'Logged out successfully']);
}

function getProfile($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT id, name, email, phone, role, status, is_on_duty, created_at, last_seen FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if ($user) {
        echo json_encode(['success' => true, 'data' => $user]);
    } else {
        echo json_encode(['success' => false, 'message' => 'User not found']);
    }
}

function updateProfile($pdo, $userId) {
    $name = sanitize($_POST['name'] ?? '');
    $phone = sanitize($_POST['phone'] ?? '');
    
    $stmt = $pdo->prepare("UPDATE users SET name = ?, phone = ? WHERE id = ?");
    $stmt->execute([$name, $phone, $userId]);
    
    echo json_encode(['success' => true, 'message' => 'Profile updated']);
}

function createCheckin($pdo, $userId) {
    $latitude = (float)$_POST['latitude'];
    $longitude = (float)$_POST['longitude'];
    $locationName = sanitize($_POST['location_name'] ?? '');
    $photoUrl = sanitize($_POST['photo_url'] ?? '');
    
    $stmt = $pdo->prepare("INSERT INTO check_ins (user_id, latitude, longitude, location_name, photo_url, status) VALUES (?, ?, ?, ?, ?, 'pending')");
    $stmt->execute([$userId, $latitude, $longitude, $locationName, $photoUrl]);
    
    $checkinId = $pdo->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'message' => 'Check-in recorded',
        'data' => ['id' => $checkinId]
    ]);
}

function getMyCheckins($pdo, $userId) {
    $limit = (int)($_GET['limit'] ?? 50);
    
    $stmt = $pdo->prepare("SELECT * FROM check_ins WHERE user_id = ? ORDER BY created_at DESC LIMIT ?");
    $stmt->execute([$userId, $limit]);
    $checkins = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $checkins]);
}

function updateLocation($pdo, $userId) {
    $latitude = (float)$_POST['latitude'];
    $longitude = (float)$_POST['longitude'];
    $accuracy = (float)($_POST['accuracy'] ?? 0);
    $speed = (float)($_POST['speed'] ?? 0);
    $bearing = (float)($_POST['bearing'] ?? 0);
    $locationName = sanitize($_POST['location_name'] ?? '');
    
    $stmt = $pdo->prepare("INSERT INTO location_logs (user_id, latitude, longitude, accuracy, speed, bearing, location_name) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([$userId, $latitude, $longitude, $accuracy, $speed, $bearing, $locationName]);
    
    // Update user's last seen
    $pdo->prepare("UPDATE users SET last_seen = NOW() WHERE id = ?")->execute([$userId]);
    
    echo json_encode(['success' => true, 'message' => 'Location updated']);
}

function getMyLocation($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT * FROM location_logs WHERE user_id = ? ORDER BY created_at DESC LIMIT 1");
    $stmt->execute([$userId]);
    $location = $stmt->fetch();
    
    echo json_encode(['success' => true, 'data' => $location]);
}

function createSOS($pdo, $userId) {
    $latitude = (float)$_POST['latitude'];
    $longitude = (float)$_POST['longitude'];
    $locationName = sanitize($_POST['location_name'] ?? '');
    $description = sanitize($_POST['description'] ?? '');
    $photoUrl = sanitize($_POST['photo_url'] ?? '');
    
    $stmt = $pdo->prepare("INSERT INTO sos_alerts (user_id, latitude, longitude, location_name, description, photo_url, status) VALUES (?, ?, ?, ?, ?, ?, 'active')");
    $stmt->execute([$userId, $latitude, $longitude, $locationName, $description, $photoUrl]);
    
    $sosId = $pdo->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'message' => 'SOS alert created',
        'data' => ['id' => $sosId]
    ]);
}

function getMySOS($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT * FROM sos_alerts WHERE user_id = ? ORDER BY created_at DESC LIMIT 10");
    $stmt->execute([$userId]);
    $alerts = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $alerts]);
}

function resolveSOS($pdo, $userId) {
    $sosId = (int)$_POST['sos_id'];
    $notes = sanitize($_POST['notes'] ?? '');
    
    $stmt = $pdo->prepare("UPDATE sos_alerts SET status = 'resolved', resolved_by = ?, resolved_at = NOW(), description = CONCAT(IFNULL(description, ''), ?) WHERE id = ?");
    $stmt->execute([$userId, "\n[Resolved by user] " . $notes, $sosId]);
    
    echo json_encode(['success' => true, 'message' => 'SOS resolved']);
}

function getAllStaff($pdo, $userId) {
    $stmt = $pdo->query("
        SELECT u.id, u.name, u.email, u.phone, u.role, u.is_on_duty, u.status,
               l.latitude, l.longitude, l.location_name, l.created_at as location_time
        FROM users u
        LEFT JOIN (
            SELECT user_id, latitude, longitude, location_name, created_at
            FROM location_logs
            WHERE id IN (SELECT MAX(id) FROM location_logs GROUP BY user_id)
        ) l ON l.user_id = u.id
        WHERE u.status = 'active'
        ORDER BY u.name
    ");
    $staff = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $staff]);
}
