<?php
/**
 * API Endpoint for Staff Locations
 * Returns real-time location data for the map
 */

require_once '../includes/config.php';

header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    $action = $_GET['action'] ?? 'all';
    
    switch ($action) {
        case 'all':
            // Get all staff with their latest locations
            $stmt = $pdo->query("
                SELECT 
                    u.id,
                    u.name,
                    u.email,
                    u.phone,
                    u.is_on_duty,
                    u.status as user_status,
                    l.latitude,
                    l.longitude,
                    l.location_name,
                    l.created_at as last_update,
                    l.accuracy,
                    (SELECT COUNT(*) FROM sos_alerts s WHERE s.user_id = u.id AND s.status = 'active') as active_sos
                FROM users u
                LEFT JOIN (
                    SELECT user_id, latitude, longitude, location_name, created_at, accuracy
                    FROM location_logs
                    WHERE id IN (
                        SELECT MAX(id) FROM location_logs GROUP BY user_id
                    )
                ) l ON l.user_id = u.id
                WHERE u.status = 'active'
                ORDER BY u.name
            ");
            $locations = $stmt->fetchAll();
            
            $data = array_map(function($loc) {
                return [
                    'id' => (int)$loc['id'],
                    'name' => $loc['name'],
                    'email' => $loc['email'],
                    'phone' => $loc['phone'],
                    'lat' => $loc['latitude'] ? (float)$loc['latitude'] : null,
                    'lng' => $loc['longitude'] ? (float)$loc['longitude'] : null,
                    'location' => $loc['location_name'],
                    'is_on_duty' => (bool)$loc['is_on_duty'],
                    'status' => $loc['active_sos'] > 0 ? 'sos' : ($loc['is_on_duty'] ? 'safe' : 'offline'),
                    'last_update' => $loc['last_update'],
                    'accuracy' => $loc['accuracy']
                ];
            }, $locations);
            
            echo json_encode(['success' => true, 'data' => $data, 'count' => count($data)]);
            break;
            
        case 'active':
            // Get only on-duty staff locations
            $stmt = $pdo->query("
                SELECT 
                    u.id,
                    u.name,
                    l.latitude,
                    l.longitude,
                    l.location_name,
                    l.created_at as last_update,
                    (SELECT COUNT(*) FROM sos_alerts s WHERE s.user_id = u.id AND s.status = 'active') as active_sos
                FROM users u
                LEFT JOIN (
                    SELECT user_id, latitude, longitude, location_name, created_at
                    FROM location_logs
                    WHERE id IN (
                        SELECT MAX(id) FROM location_logs GROUP BY user_id
                    )
                ) l ON l.user_id = u.id
                WHERE u.is_on_duty = 1 AND u.status = 'active'
                ORDER BY u.name
            ");
            $locations = $stmt->fetchAll();
            
            $data = array_map(function($loc) {
                return [
                    'id' => (int)$loc['id'],
                    'name' => $loc['name'],
                    'lat' => $loc['latitude'] ? (float)$loc['latitude'] : null,
                    'lng' => $loc['longitude'] ? (float)$loc['longitude'] : null,
                    'location' => $loc['location_name'],
                    'status' => $loc['active_sos'] > 0 ? 'sos' : 'safe',
                    'last_update' => $loc['last_update']
                ];
            }, $locations);
            
            echo json_encode(['success' => true, 'data' => $data]);
            break;
            
        case 'sos':
            // Get staff with active SOS
            $stmt = $pdo->query("
                SELECT 
                    u.id,
                    u.name,
                    s.latitude,
                    s.longitude,
                    s.location_name,
                    s.created_at as sos_time,
                    s.description
                FROM sos_alerts s
                JOIN users u ON s.user_id = u.id
                WHERE s.status = 'active'
                ORDER BY s.created_at DESC
            ");
            $locations = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'data' => $locations]);
            break;
            
        case 'user':
            // Get specific user's location history
            $userId = (int)$_GET['user_id'];
            $limit = (int)($_GET['limit'] ?? 50);
            
            $stmt = $pdo->prepare("
                SELECT latitude, longitude, location_name, accuracy, created_at
                FROM location_logs
                WHERE user_id = ?
                ORDER BY created_at DESC
                LIMIT ?
            ");
            $stmt->execute([$userId, $limit]);
            $history = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'data' => $history]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
