<?php
/**
 * API Endpoint for Exporting Data
 * Supports CSV, JSON, and PDF exports
 */

require_once '../includes/config.php';

try {
    $type = $_GET['type'] ?? 'checkins';
    $format = $_GET['format'] ?? 'csv';
    $startDate = $_GET['start'] ?? date('Y-m-d', strtotime('-30 days'));
    $endDate = $_GET['end'] ?? date('Y-m-d');
    
    $pdo = getDBConnection();
    
    switch ($type) {
        case 'checkins':
            exportCheckins($pdo, $format, $startDate, $endDate);
            break;
        case 'sos':
            exportSOS($pdo, $format, $startDate, $endDate);
            break;
        case 'report':
            exportReport($pdo, $format, $startDate, $endDate);
            break;
        default:
            die('Invalid export type');
    }
    
} catch (Exception $e) {
    die('Export error: ' . $e->getMessage());
}

function exportCheckins($pdo, $format, $startDate, $endDate) {
    $stmt = $pdo->prepare("
        SELECT c.*, u.name as user_name, u.email as user_email, u.phone as user_phone
        FROM check_ins c
        JOIN users u ON c.user_id = u.id
        WHERE DATE(c.created_at) BETWEEN ? AND ?
        ORDER BY c.created_at DESC
    ");
    $stmt->execute([$startDate, $endDate]);
    $data = $stmt->fetchAll();
    
    if ($format === 'json') {
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="checkins_' . $startDate . '_' . $endDate . '.json"');
        echo json_encode($data, JSON_PRETTY_PRINT);
    } else {
        // CSV
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="checkins_' . $startDate . '_' . $endDate . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Header
        fputcsv($output, ['ID', 'Staff Name', 'Email', 'Phone', 'Location', 'Latitude', 'Longitude', 'Status', 'Created At', 'Verified At']);
        
        // Data
        foreach ($data as $row) {
            fputcsv($output, [
                $row['id'],
                $row['user_name'],
                $row['user_email'],
                $row['user_phone'],
                $row['location_name'],
                $row['latitude'],
                $row['longitude'],
                $row['status'],
                $row['created_at'],
                $row['verified_at'] ?? ''
            ]);
        }
        
        fclose($output);
    }
    exit;
}

function exportSOS($pdo, $format, $startDate, $endDate) {
    $stmt = $pdo->prepare("
        SELECT s.*, u.name as user_name, u.email as user_email, u.phone as user_phone
        FROM sos_alerts s
        JOIN users u ON s.user_id = u.id
        WHERE DATE(s.created_at) BETWEEN ? AND ?
        ORDER BY s.created_at DESC
    ");
    $stmt->execute([$startDate, $endDate]);
    $data = $stmt->fetchAll();
    
    if ($format === 'json') {
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="sos_alerts_' . $startDate . '_' . $endDate . '.json"');
        echo json_encode($data, JSON_PRETTY_PRINT);
    } else {
        // CSV
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="sos_alerts_' . $startDate . '_' . $endDate . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Header
        fputcsv($output, ['ID', 'Staff Name', 'Email', 'Phone', 'Location', 'Latitude', 'Longitude', 'Status', 'Description', 'Created At', 'Resolved At']);
        
        // Data
        foreach ($data as $row) {
            fputcsv($output, [
                $row['id'],
                $row['user_name'],
                $row['user_email'],
                $row['user_phone'],
                $row['location_name'],
                $row['latitude'],
                $row['longitude'],
                $row['status'],
                $row['description'],
                $row['created_at'],
                $row['resolved_at'] ?? ''
            ]);
        }
        
        fclose($output);
    }
    exit;
}

function exportReport($pdo, $format, $startDate, $endDate) {
    // Get summary stats
    $checkinStats = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'verified' THEN 1 ELSE 0 END) as verified,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
        FROM check_ins 
        WHERE DATE(created_at) BETWEEN ? AND ?
    ");
    $checkinStats->execute([$startDate, $endDate]);
    $checkinData = $checkinStats->fetch();
    
    $sosStats = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'acknowledged' THEN 1 ELSE 0 END) as acknowledged,
            SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved
        FROM sos_alerts 
        WHERE DATE(created_at) BETWEEN ? AND ?
    ");
    $sosStats->execute([$startDate, $endDate]);
    $sosData = $sosStats->fetch();
    
    // Get staff activity
    $staffActivity = $pdo->prepare("
        SELECT 
            u.name,
            COUNT(DISTINCT c.id) as checkin_count,
            COUNT(DISTINCT s.id) as sos_count
        FROM users u
        LEFT JOIN check_ins c ON c.user_id = u.id AND DATE(c.created_at) BETWEEN ? AND ?
        LEFT JOIN sos_alerts s ON s.user_id = u.id AND DATE(s.created_at) BETWEEN ? AND ?
        WHERE u.status = 'active'
        GROUP BY u.id
        ORDER BY checkin_count DESC
    ");
    $staffActivity->execute([$startDate, $endDate, $startDate, $endDate]);
    $staffData = $staffActivity->fetchAll();
    
    $report = [
        'period' => ['start' => $startDate, 'end' => $endDate],
        'summary' => [
            'checkins' => $checkinData,
            'sos' => $sosData
        ],
        'staff_activity' => $staffData
    ];
    
    if ($format === 'json') {
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="report_' . $startDate . '_' . $endDate . '.json"');
        echo json_encode($report, JSON_PRETTY_PRINT);
    } else {
        // CSV
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="report_' . $startDate . '_' . $endDate . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Summary
        fputcsv($output, ['Report Summary']);
        fputcsv($output, ['Period', $startDate . ' to ' . $endDate]);
        fputcsv($output, []);
        fputcsv($output, ['Check-in Statistics']);
        fputcsv($output, ['Total Check-ins', $checkinData['total']]);
        fputcsv($output, ['Verified', $checkinData['verified']]);
        fputcsv($output, ['Pending', $checkinData['pending']]);
        fputcsv($output, ['Failed', $checkinData['failed']]);
        fputcsv($output, []);
        fputcsv($output, ['SOS Alert Statistics']);
        fputcsv($output, ['Total Alerts', $sosData['total']]);
        fputcsv($output, ['Active', $sosData['active']]);
        fputcsv($output, ['Acknowledged', $sosData['acknowledged']]);
        fputcsv($output, ['Resolved', $sosData['resolved']]);
        fputcsv($output, []);
        fputcsv($output, ['Staff Activity']);
        fputcsv($output, ['Staff Name', 'Check-ins', 'SOS Alerts']);
        
        foreach ($staffData as $row) {
            fputcsv($output, [$row['name'], $row['checkin_count'], $row['sos_count']]);
        }
        
        fclose($output);
    }
    exit;
}
